// BackChat Chrome Extension - Popup Script
const API_BASE_URL = 'https://backchat.kurasika.tech';

// State management
let currentUser = null;
let sessionState = 'menu';
let isProcessing = false;

// DOM Elements
const loginSection = document.getElementById('loginSection');
const chatSection = document.getElementById('chatSection');
const loginPhone = document.getElementById('loginPhone');
const loginButton = document.getElementById('loginButton');
const loginError = document.getElementById('loginError');
const messagesContainer = document.getElementById('messagesContainer');
const messageInput = document.getElementById('messageInput');
const sendButton = document.getElementById('sendButton');
const loadingOverlay = document.getElementById('loadingOverlay');
const userPhone = document.getElementById('userPhone');
const userBalance = document.getElementById('userBalance');

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
    // Check if user is already logged in
    const savedUser = await getStoredUser();
    if (savedUser) {
        currentUser = savedUser;
        showChatSection();
        sendMessage('menu'); // Load main menu
    }
    
    // Event listeners
    loginButton.addEventListener('click', handleLogin);
    loginPhone.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') handleLogin();
    });
    
    sendButton.addEventListener('click', handleSendMessage);
    messageInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            handleSendMessage();
        }
    });
});

// Storage functions
async function getStoredUser() {
    return new Promise((resolve) => {
        chrome.storage.local.get(['user'], (result) => {
            resolve(result.user || null);
        });
    });
}

async function storeUser(user) {
    return new Promise((resolve) => {
        chrome.storage.local.set({ user }, resolve);
    });
}

async function clearStoredUser() {
    return new Promise((resolve) => {
        chrome.storage.local.remove(['user'], resolve);
    });
}

// Login handler
async function handleLogin() {
    const phone = loginPhone.value.trim();
    
    if (!phone) {
        loginError.textContent = 'Please enter your phone number';
        return;
    }
    
    if (!phone.startsWith('+')) {
        loginError.textContent = 'Phone number must start with +';
        return;
    }
    
    setLoading(true);
    loginError.textContent = '';
    
    try {
        const response = await fetch(`${API_BASE_URL}/api/extension/login`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ phone_number: phone })
        });
        
        const data = await response.json();
        
        if (response.ok && data.success) {
            currentUser = data.user;
            await storeUser(currentUser);
            showChatSection();
            
            // Send initial greeting
            sendMessage('menu');
        } else {
            loginError.textContent = data.message || 'Login failed. Please try again.';
        }
    } catch (error) {
        console.error('Login error:', error);
        loginError.textContent = 'Connection error. Please check your internet.';
    } finally {
        setLoading(false);
    }
}

// UI functions
function showChatSection() {
    loginSection.style.display = 'none';
    chatSection.style.display = 'flex';
    
    if (currentUser) {
        userPhone.textContent = currentUser.whatsapp_number || currentUser.phone_number;
        userBalance.textContent = `$${parseFloat(currentUser.balance).toFixed(2)}`;
    }
    
    messageInput.focus();
}

function setLoading(loading) {
    loadingOverlay.style.display = loading ? 'flex' : 'none';
    loginButton.disabled = loading;
    sendButton.disabled = loading;
    messageInput.disabled = loading;
}

// Message handling
async function handleSendMessage() {
    const message = messageInput.value.trim();
    
    if (!message || isProcessing) return;
    
    // Add user message to UI
    addMessage(message, 'user');
    messageInput.value = '';
    
    // Send to server
    await sendMessage(message);
}

async function sendMessage(message) {
    if (isProcessing) return;
    
    isProcessing = true;
    setLoading(true);
    
    try {
        const response = await fetch(`${API_BASE_URL}/api/extension/message`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                user_id: currentUser.id,
                message: message,
                phone_number: currentUser.whatsapp_number || currentUser.phone_number
            })
        });
        
        const data = await response.json();
        
        if (response.ok && data.success) {
            // Update user balance if provided
            if (data.user) {
                currentUser.balance = data.user.balance;
                userBalance.textContent = `$${parseFloat(currentUser.balance).toFixed(2)}`;
                await storeUser(currentUser);
            }
            
            // Add bot response
            if (data.response && data.response.message) {
                addMessage(data.response.message, 'bot');
            }
            
            // Handle special message types
            if (data.response && data.response.type === 'payment_link') {
                // Open payment link in new tab
                if (data.response.payment_url) {
                    chrome.tabs.create({ url: data.response.payment_url });
                }
            }
            
            // Update session state
            if (data.session_state) {
                sessionState = data.session_state;
            }
        } else {
            addMessage(data.message || 'Sorry, something went wrong. Please try again.', 'bot');
        }
    } catch (error) {
        console.error('Message error:', error);
        addMessage('Connection error. Please check your internet and try again.', 'bot');
    } finally {
        isProcessing = false;
        setLoading(false);
        messageInput.focus();
    }
}

// Add message to UI
function addMessage(text, type) {
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${type}`;
    
    const bubble = document.createElement('div');
    bubble.className = 'message-bubble';
    
    const textDiv = document.createElement('div');
    textDiv.className = 'message-text';
    
    // Format message text (convert markdown-style formatting)
    let formattedText = text
        .replace(/`([^`]+)`/g, '<code>$1</code>')
        .replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>')
        .replace(/\*([^*]+)\*/g, '<strong>$1</strong>')
        .replace(/_([^_]+)_/g, '<em>$1</em>');
    
    textDiv.innerHTML = formattedText;
    
    const timeDiv = document.createElement('div');
    timeDiv.className = 'message-time';
    timeDiv.textContent = new Date().toLocaleTimeString('en-US', { 
        hour: '2-digit', 
        minute: '2-digit' 
    });
    
    bubble.appendChild(textDiv);
    bubble.appendChild(timeDiv);
    messageDiv.appendChild(bubble);
    
    messagesContainer.appendChild(messageDiv);
    
    // Scroll to bottom
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

// Notification function
function showNotification(title, message) {
    if (Notification.permission === 'granted') {
        chrome.notifications.create({
            type: 'basic',
            iconUrl: 'icons/icon128.png',
            title: title,
            message: message
        });
    }
}

// Poll for updates (check for call status updates)
setInterval(async () => {
    if (!currentUser || !chatSection || chatSection.style.display === 'none') return;
    
    try {
        const response = await fetch(`${API_BASE_URL}/api/extension/status`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                user_id: currentUser.id
            })
        });
        
        const data = await response.json();
        
        if (data.success && data.updates) {
            // Update balance
            if (data.user && data.user.balance !== currentUser.balance) {
                currentUser.balance = data.user.balance;
                userBalance.textContent = `$${parseFloat(currentUser.balance).toFixed(2)}`;
                await storeUser(currentUser);
            }
            
            // Show any pending messages
            if (data.updates.messages && data.updates.messages.length > 0) {
                data.updates.messages.forEach(msg => {
                    addMessage(msg.message, 'bot');
                    if (msg.notification) {
                        showNotification('BackChat', msg.message);
                    }
                });
            }
        }
    } catch (error) {
        console.error('Status check error:', error);
    }
}, 5000); // Check every 5 seconds

// Request notification permission on load
if (Notification.permission === 'default') {
    Notification.requestPermission();
}
